
const api = chrome || browser;

let state = {
  enabled: false,
  mode: "manual", // 'manual' | 'token'
  manual: {
    scheme: "http",
    host: "",
    port: "",
    username: "",
    password: "",
    bypass: "<local>"
  },
  tokenCfg: null,   // { token, apiBase }
  lastList: [],     // last proxies loaded from /proxy/list
  currentProxy: null // { scheme, host, port, username, password, bypass }
};

function log(...args){
  try { console.log("[ProxySell]", ...args); } catch(e){}
}

async function loadState(){
  return new Promise(resolve => {
    try{
      api.storage.local.get(["state"], res => {
        if (res && res.state){
          state = Object.assign({}, state, res.state);
        }
        resolve(state);
      });
    }catch(e){
      log("loadState error", e);
      resolve(state);
    }
  });
}

async function saveState(){
  return new Promise(resolve => {
    try{
      api.storage.local.set({ state }, () => resolve(true));
    }catch(e){
      log("saveState error", e);
      resolve(false);
    }
  });
}

function buildProxyConfig(p){
  const scheme = p.scheme || "http";
  const host   = p.host;
  const port   = Number(p.port) || 0;
  const bypass = (p.bypass || "<local>").split(",").map(s => s.trim()).filter(Boolean);

  if (!host || !port){
    return null;
  }

  return {
    mode: "fixed_servers",
    rules: {
      singleProxy: { scheme, host, port },
      bypassList: bypass
    }
  };
}

function setChromeProxy(config){
  return new Promise(resolve => {
    if (!api.proxy || !api.proxy.settings){
      resolve({ ok:false, error:"Proxy API not available" });
      return;
    }
    api.proxy.settings.set(
      { value: config || { mode:"system" }, scope: "regular" },
      () => {
        if (api.runtime.lastError){
          resolve({ ok:false, error: api.runtime.lastError.message });
        }else{
          resolve({ ok:true });
        }
      }
    );
  });
}

async function applyProxy(p){
  const cfg = buildProxyConfig(p);
  if (!cfg){
    return { ok:false, error:"Invalid proxy config" };
  }
  const r = await setChromeProxy(cfg);
  if (r.ok){
    state.enabled = true;
    state.currentProxy = {
      scheme: p.scheme || "http",
      host: p.host,
      port: Number(p.port) || 0,
      username: p.username || "",
      password: p.password || "",
      bypass: p.bypass || "<local>"
    };
    await saveState();
  }
  return r;
}

async function disableProxy(){
  const r = await setChromeProxy({ mode:"system" });
  if (r.ok){
    state.enabled = false;
    state.currentProxy = null;
    await saveState();
  }
  return r;
}

api.runtime.onMessage.addListener((msg, sender, sendResponse) => {
  (async () => {
    await loadState();
    let res = { ok:false, error:"Unknown action" };

    try{
      switch(msg.type){

        case "getState":
        case "read":
          res = { ok:true, state };
          break;

        case "saveTokenCfg":
          state.tokenCfg = msg.tokenCfg || null;
          await saveState();
          res = { ok:true };
          break;

        case "saveList":
          state.lastList = Array.isArray(msg.list) ? msg.list : [];
          await saveState();
          res = { ok:true };
          break;

        case "applyManual":
          state.mode = "manual";
          state.manual = Object.assign({}, state.manual, msg.manual || {});
          res = await applyProxy(state.manual);
          break;

        case "applyTokenProxy":
          state.mode = "token";
          res = await applyProxy(msg.proxy || {});
          break;

        case "enable":
          if (state.currentProxy){
            res = await applyProxy(state.currentProxy);
          } else if (state.mode === "manual" && state.manual && state.manual.host){
            res = await applyProxy(state.manual);
          } else {
            res = { ok:false, error:"No proxy selected" };
          }
          break;

        case "disable":
          res = await disableProxy();
          break;
      }
    }catch(e){
      log("onMessage error", e);
      res = { ok:false, error:String(e) };
    }

    sendResponse(res);
  })();
  return true;
});

// init
loadState().then(s => {
  log("Service worker started with state", s);
});

// Stubs for compatibility (no-op on MV3 public)
function ensureAuthListener(){ return; }
function removeAuthListener(){ return; }
