// ===== Helpers =====
const api = chrome || browser;
const qs  = id => document.getElementById(id);

function setStatus(id, text) {
    const el = qs(id);
    if (!el) return;
    el.textContent = text || "";
    if (text) {
        setTimeout(() => {
            if (el.textContent === text) el.textContent = "";
        }, 3000);
    }
}

function show(id) { qs(id).style.display = ""; }
function hide(id) { qs(id).style.display = "none"; }

function send(msg) {
    return new Promise(resolve => {
        try {
            api.runtime.sendMessage(msg, resp => resolve(resp || {}));
        } catch (e) {
            resolve({ ok: false, error: String(e) });
        }
    });
}

// ===== Snapshot / withProxyDisabled =====

// Lấy state hiện tại xem có proxy nào đang bật không
async function snapshotCurrentProxy() {
    const stResp = await send({ type: "getState" });
    if (!stResp || !stResp.ok || !stResp.state) return null;
    const st = stResp.state;
    if (!st.enabled || !st.currentProxy) return null;
    return Object.assign({}, st.currentProxy); // clone
}

// Tạm disable proxy khi chạy fn (xoay), xong restore lại nếu trước đó đang bật
async function withProxyDisabled(fn) {
    const prev = await snapshotCurrentProxy();
    if (prev) {
        await send({ type: "disable" });
    }
    try {
        return await fn();
    } finally {
        if (prev) {
            await send({ type: "applyTokenProxy", proxy: prev });
        }
    }
}

// ===== Tab switch =====
qs("tabManual").onclick = () => {
    qs("tabManual").classList.add("active");
    qs("tabToken").classList.remove("active");
    show("viewManual");
    hide("viewToken");
};

qs("tabToken").onclick = () => {
    qs("tabToken").classList.add("active");
    qs("tabManual").classList.remove("active");
    hide("viewManual");
    show("viewToken");
};

// ===== MANUAL =====
qs("m_use").onclick = async () => {
    const manual = {
        scheme: qs("m_scheme").value.trim() || "http",
        host: qs("m_host").value.trim(),
        port: Number(qs("m_port").value.trim() || 0),
        username: qs("m_user").value.trim(),
        password: qs("m_pass").value.trim(),
        bypass: qs("m_bypass").value.trim() || "<local>",
    };

    const r = await send({ type: "applyManual", manual });
    setStatus("statusManual", r.ok ? "Applied" : (r.error || "Failed"));
};

qs("m_disable").onclick = async () => {
    const r = await send({ type: "disable" });
    setStatus("statusManual", r.ok ? "Disabled" : (r.error || "Failed"));
};

// ===== TOKEN: LOAD LIST =====
qs("t_load").onclick = async () => {
    const tokenRaw = qs("t_token").value.trim();
    if (!tokenRaw) {
        setStatus("statusToken", "Token required");
        return;
    }

    let apiBase = qs("t_apiBase").value.trim();
    if (!apiBase) apiBase = "https://api.proxysell.net";
    apiBase = apiBase.replace(/\/$/, "");

    await send({ type: "saveTokenCfg", tokenCfg: { token: tokenRaw, apiBase } });

    try {
        const res = await fetch(apiBase + "/proxy/list", {
            method: "POST",
            headers: {
                Authorization: tokenRaw,
                "Content-Type": "application/json",
            },
            body: JSON.stringify({}),
        });

        if (!res.ok) throw new Error("HTTP " + res.status);

        const list = await res.json();
        await send({ type: "saveList", list });

        const stResp = await send({ type: "getState" });
        const currentProxy =
            stResp && stResp.ok && stResp.state ? stResp.state.currentProxy : null;

        renderProxyList(list, currentProxy);
        setStatus("statusToken", "Loaded " + list.length);
    } catch (err) {
        console.error(err);
        setStatus("statusToken", "Load failed: " + err.message);
    }
};

// ===== TOKEN: ROTATE ALL =====
qs("t_rotateAll").onclick = async () => {
    const tokenRaw = qs("t_token").value.trim();
    if (!tokenRaw) {
        setStatus("statusToken", "Token required");
        return;
    }

    let apiBase = qs("t_apiBase").value.trim();
    if (!apiBase) apiBase = "https://api.proxysell.net";
    apiBase = apiBase.replace(/\/$/, "");

    const items = document.querySelectorAll(".proxy-item[data-order-id]");
    const orderIds = new Set();

    items.forEach((row) => {
        const type = Number(row.getAttribute("data-proxy-type") || 0);
        if (type === 1) {
            const oid = row.getAttribute("data-order-id");
            if (oid) orderIds.add(oid);
        }
    });

    if (!orderIds.size) {
        setStatus("statusToken", "No rotating proxy found");
        return;
    }

    await withProxyDisabled(async () => {
        let ok = 0,
            fail = 0;

        for (const oid of orderIds) {
            try {
                const res = await fetch(apiBase + "/proxy/rotate", {
                    method: "POST",
                    headers: {
                        Authorization: tokenRaw,
                        "Content-Type": "application/json",
                    },
                    body: JSON.stringify({ order_id: Number(oid) || oid }),
                });
                if (res.ok) ok++;
                else fail++;
            } catch (e) {
                fail++;
            }
        }

        setStatus(
            "statusToken",
            `Rotate all done: ${ok} ok / ${fail} fail`
        );
    });
};

// ===== TOKEN: Enable / Disable =====
qs("t_enable").onclick = async () => {
    const r = await send({ type: "enable" });
    setStatus("statusToken", r.ok ? "Enabled" : (r.error || "Failed"));
};

qs("t_disable").onclick = async () => {
    const r = await send({ type: "disable" });
    setStatus("statusToken", r.ok ? "Disabled" : (r.error || "Failed"));
};

// ===== RENDER LIST =====
function renderProxyList(list, currentProxy) {
    const box = qs("proxyList");
    box.innerHTML = "";

    const curHost = currentProxy && (currentProxy.host || currentProxy.ip);
    const curPort = currentProxy && Number(currentProxy.port);

    (list || []).forEach((p) => {
        const host = p.ip || p.host || "";
        const port = Number(p.port || 0);
        const scheme =
            p.protocol || p.scheme || (p.type && p.type.includes("socks") ? "socks5" : "http");
        const orderId = p.order_id || p.orderId || "";
        const proxyType = Number(p.proxyType || p.proxy_type || 0); // 0 static, 1 rotate

        const row = document.createElement("div");
        row.className = "proxy-item";
        row.setAttribute("data-ip", host);
        row.setAttribute("data-port", String(port));
        if (orderId) row.setAttribute("data-order-id", orderId);
        row.setAttribute("data-proxy-type", String(proxyType));

        row.innerHTML = `
            <div class="proxy-meta">
                <div><strong>${host}:${port}</strong> <span class="small">(${scheme})</span></div>
                <div class="small">Order #${orderId || "-"}</div>
            </div>
            <div class="proxy-actions">
                <button class="btn primary">Use</button>
                ${
            proxyType === 1
                ? '<button class="btn ghost js-rotate">Rotate</button>'
                : ""
        }
                <span class="badge js-using" style="display:none">Using</span>
            </div>
        `;

        const useBtn = row.querySelector(".btn.primary");
        const rotateBtn = row.querySelector(".js-rotate");
        const usingBadge = row.querySelector(".js-using");

        // Đánh dấu Using nếu trùng currentProxy
        if (curHost && curPort && host === curHost && port === curPort) {
            usingBadge.style.display = "";
        }

        // ----- Use -----
        useBtn.onclick = async () => {
            const tokenRaw = qs("t_token").value.trim();
            let apiBase = qs("t_apiBase").value.trim();
            if (!apiBase) apiBase = "https://api.proxysell.net";
            apiBase = apiBase.replace(/\/$/, "");

            await send({
                type: "saveTokenCfg",
                tokenCfg: { token: tokenRaw, apiBase },
            });

            const proxy = {
                scheme,
                host,
                port,
                username: p.username || "",
                password: p.password || "",
                bypass: "<local>",
            };

            const r = await send({ type: "applyTokenProxy", proxy });
            if (r.ok) {
                document
                    .querySelectorAll(".js-using")
                    .forEach((el) => (el.style.display = "none"));
                usingBadge.style.display = "";
                setStatus("statusToken", "Applied");
            } else {
                setStatus("statusToken", r.error || "Apply failed");
            }
        };

        // ----- Rotate 1 (chỉ proxyType=1) -----
        if (rotateBtn) {
            rotateBtn.onclick = async () => {
                const tokenRaw = qs("t_token").value.trim();
                if (!tokenRaw) {
                    setStatus("statusToken", "Token required");
                    return;
                }

                let apiBase = qs("t_apiBase").value.trim();
                if (!apiBase) apiBase = "https://api.proxysell.net";
                apiBase = apiBase.replace(/\/$/, "");

                if (!orderId) {
                    setStatus("statusToken", "Missing order_id");
                    return;
                }

                try {
                    await withProxyDisabled(async () => {
                        const body = {
                            order_id: Number(orderId) || orderId,
                            port: port,
                        };

                        const res = await fetch(apiBase + "/proxy/rotate", {
                            method: "POST",
                            headers: {
                                Authorization: tokenRaw,
                                "Content-Type": "application/json",
                            },
                            body: JSON.stringify(body),
                        });

                        if (!res.ok) {
                            throw new Error("HTTP " + res.status);
                        }

                        setStatus("statusToken", "Rotated");
                    });
                } catch (err) {
                    setStatus(
                        "statusToken",
                        "Rotate failed: " + err.message
                    );
                }
            };
        }

        box.appendChild(row);
    });
}

// ===== INIT =====
(async function init() {
    const res = await send({ type: "getState" });

    if (res && res.ok && res.state) {
        const st = res.state;

        // Manual
        const m = st.manual || {};
        if (m.scheme) qs("m_scheme").value = m.scheme;
        qs("m_host").value = m.host || "";
        qs("m_port").value = m.port || "";
        qs("m_user").value = m.username || "";
        qs("m_pass").value = m.password || "";
        qs("m_bypass").value = m.bypass || "<local>";

        // Token config
        const t = st.tokenCfg || {};
        qs("t_apiBase").value = t.apiBase || "https://api.proxysell.net";
        qs("t_token").value = t.token || "";

        // List + Using
        if (Array.isArray(st.lastList) && st.lastList.length) {
            renderProxyList(st.lastList, st.currentProxy || null);
        }
    } else {
        qs("t_apiBase").value = "https://api.proxysell.net";
        qs("m_bypass").value = "<local>";
    }
})();
