const send = (msg) => new Promise(resolve => chrome.runtime.sendMessage(msg, (resp) => resolve(resp)));
const setStatus = (t) => document.getElementById('status').innerText = t || '';
const setError  = (t) => document.getElementById('error').innerText  = t || '';

function showTab(which){
  const manual = document.getElementById('manualArea');
  const token  = document.getElementById('tokenArea');
  const tMan   = document.getElementById('tab-manual');
  const tTok   = document.getElementById('tab-token');
  if (which === 'manual'){
    manual.style.display = ''; token.style.display = 'none';
    tMan.classList.add('active'); tTok.classList.remove('active');
  } else {
    manual.style.display = 'none'; token.style.display = '';
    tTok.classList.add('active'); tMan.classList.remove('active');
  }
}
document.getElementById('tab-manual').addEventListener('click', ()=> showTab('manual'));
document.getElementById('tab-token').addEventListener('click',  ()=> showTab('token'));

function updateFooter(proxy){
  const el = document.getElementById('footerProxy');
  if (!el) return;
  if (!proxy || !proxy.host) el.innerText = 'none';
  else el.innerText = `${proxy.host}:${proxy.port}`;
}
function normalizeScheme(x){
  if (!x) return 'http';
  const s = String(x).toLowerCase().trim();
  if (s==='sock5' || s==='socks' || s==='socks5h') return 'socks5';
  if (s==='socks4a') return 'socks4';
  return (['http','https','socks4','socks5'].includes(s)) ? s : 'http';
}
function markCurrent(host, port){
  const items = Array.from(document.querySelectorAll('.proxy-item'));
  items.forEach(it=>{
    it.style.outline=''; const meta = it.querySelector('.proxy-meta'); if (!meta) return;
    if (meta.dataset.host===String(host) && meta.dataset.port===String(port)){
      it.style.outline='2px solid #dbeafe';
      const dot = it.querySelector('.current-dot'); if (dot) dot.style.visibility='visible';
      const useBtn = it.querySelector('.btn.primary'); if (useBtn){ useBtn.disabled=true; useBtn.textContent='Using'; }
    }
  });
}
function renderProxyList(list=[], currentProxy){
  const wrap = document.getElementById('proxyList');
  wrap.innerHTML='';
  (list||[]).forEach(p=>{
    const item = document.createElement('div'); item.className='proxy-item';
    const left = document.createElement('div'); left.className='proxy-meta';
    const host = p.ip || p.host || p.hostname || ''; const port = p.port || p.p || '';
    const scheme = normalizeScheme(p.protocol || p.scheme || (p.type && (p.type.includes('socks')?'socks5':'http')) || 'http');
    left.dataset.host=String(host); left.dataset.port=String(port);
    left.innerHTML = `<div style="display:flex;align-items:center;gap:8px">
      <span class="current-dot" style="visibility:hidden;width:8px;height:8px;border-radius:50%;background:#0b5cff;display:inline-block"></span>
      <strong>${host}:${port}</strong> <small style="color:#666">(${scheme})</small></div>
      <div style="font-size:12px;color:#666">${p.country||''} ${p.note||''}</div>`;

    const right = document.createElement('div'); right.style.display='flex'; right.style.gap='6px';

    const useBtn = document.createElement('button'); useBtn.className='btn small primary'; useBtn.innerText='Use';
    useBtn.onclick = async ()=>{
      const proxyObj = { scheme, host, port:Number(port), username:p.username||'', password:p.password||'', bypassList:['<local>'] };
      const token = (document.getElementById('token')?.value||'').trim();
      const apiBase = (document.getElementById('apiBase')?.value||'').trim();
      const cfgPatch = { enabled:true, mode:'token', _lastFetched:proxyObj, _currentProxy:proxyObj };
      if (token) cfgPatch.token = token; if (apiBase) cfgPatch.apiBase = apiBase;
      await send({type:'setConfig', config: cfgPatch});
      const resp = await send({type:'applyConfigNow'});
      if (resp?.ok){ setStatus(`Applied ${host}:${port} (enabled)`); markCurrent(host,port); updateFooter(proxyObj); }
      else setError(resp?.error || 'Apply failed');
    };
    right.appendChild(useBtn);

    const rotBtn = document.createElement('button'); rotBtn.className='btn small ghost'; rotBtn.innerText='Rotate';
    rotBtn.onclick = async ()=>{
      try{
        const token = (document.getElementById('token')?.value||'').trim();
        const apiBase = (document.getElementById('apiBase')?.value||'').trim().replace(/\/$/,'');
        if (!token) return setError('Token required');
        const res = await fetch(apiBase + '/server/proxy/rotate', {
          method:'POST', headers:{ Authorization: token, 'Content-Type':'application/json' },
          body: JSON.stringify({ ip: host, port: Number(port) })
        });
        if (!res.ok) throw new Error('Rotate failed: '+res.status);
        setStatus('Rotate OK for '+host+':'+port);
        document.getElementById('loadList')?.click();
      }catch(err){ setError(err.message || String(err)); }
    };
    right.appendChild(rotBtn);

    const tabBtn = document.createElement('button'); tabBtn.className='btn small ghost'; tabBtn.innerText='Tab';
    tabBtn.title='Apply to current tab (tab-scoped)';
    tabBtn.onclick = async ()=>{
      const tab = await new Promise(r => chrome.tabs.query({active:true,lastFocusedWindow:true}, t => r(t[0])));
      if (!tab) return setError('Không tìm thấy tab');
      const proxyObj = { scheme, host, port:Number(port), username:p.username||'', password:p.password||'', bypassList:['<local>'] };
      const resp = await send({type:'applyProxyToTab', tabId: tab.id, proxy: proxyObj});
      if (resp?.ok){ setStatus('Applied to tab '+tab.id+' (enabled)'); updateFooter(proxyObj); }
      else setError(resp?.error || 'Apply failed');
    };
    right.appendChild(tabBtn);

    item.appendChild(left); item.appendChild(right); wrap.appendChild(item);
  });

  const curHost = (currentProxy && (currentProxy.host || currentProxy.ip)) || '';
  const curPort = (currentProxy && currentProxy.port) || '';
  if (list && list.length && curHost && curPort) { markCurrent(curHost, curPort); updateFooter(currentProxy); }
}

async function loadConfigToUI(){
  const r = await send({type:'getConfig'}); const cfg = (r && r.config) || {};
  showTab(cfg.mode === 'token' ? 'token' : 'manual');
  const m = cfg.manual || {};
  document.getElementById('scheme').value = m.scheme || 'http';
  document.getElementById('host').value   = m.host || '';
  document.getElementById('port').value   = m.port || '';
  document.getElementById('bypass').value = (m.bypassList||[]).join(',');
  document.getElementById('username').value = m.username||'';
  document.getElementById('password').value = m.password||'';
  document.getElementById('token').value    = cfg.token||'';
  document.getElementById('apiBase').value  = cfg.apiBase||'https://api.proxysell.net';

  if (cfg._proxyList) renderProxyList(cfg._proxyList, cfg._currentProxy);
  updateFooter(cfg._currentProxy || cfg.manual);
}

document.getElementById('applyManual').addEventListener('click', async ()=>{
  setError('');
  const cfg = {
    enabled: true,
    mode: 'manual',
    manual:{
      scheme: document.getElementById('scheme').value,
      host: document.getElementById('host').value,
      port: Number(document.getElementById('port').value)||0,
      username: document.getElementById('username').value||'',
      password: document.getElementById('password').value||'',
      bypassList: (document.getElementById('bypass').value||'').split(',').map(s=>s.trim()).filter(Boolean)
    }
  };
  await send({type:'setConfig', config: cfg});
  const resp = await send({type:'applyConfigNow'});
  if (resp?.ok){ setStatus('Applied manual (enabled)'); updateFooter(cfg.manual); }
  else setError(resp?.error || 'Apply failed');
});

document.getElementById('disableProxy').addEventListener('click', async ()=>{
  await send({type:'setConfig', config:{enabled:false}});
  await send({type:'applyConfigNow'});
  setStatus('Proxy disabled'); updateFooter(null);
});

document.getElementById('applyToTab').addEventListener('click', async ()=>{
  const tab = await new Promise(r => chrome.tabs.query({active:true,lastFocusedWindow:true}, t => r(t[0])));
  if (!tab) return setError('Không tìm thấy tab hiện tại');
  const proxyObj = {
    scheme: document.getElementById('scheme').value,
    host: document.getElementById('host').value,
    port: Number(document.getElementById('port').value)||0,
    username: document.getElementById('username').value||'',
    password: document.getElementById('password').value||'',
    bypassList: (document.getElementById('bypass').value||'').split(',').map(s=>s.trim()).filter(Boolean)
  };
  const resp = await send({type:'applyProxyToTab', tabId: tab.id, proxy: proxyObj});
  if (resp?.ok){ setStatus('Applied to tab '+tab.id+' (enabled)'); updateFooter(proxyObj); }
  else setError(resp?.error || 'Apply to tab failed');
});

document.getElementById('loadList').addEventListener('click', async ()=>{
  setError(''); setStatus('Loading list...');
  const token = document.getElementById('token').value.trim();
  const apiBase = document.getElementById('apiBase').value.trim().replace(/\/$/,'');
  if (!token) return setError('Token required');
  try{
    const resp = await fetch(apiBase + '/server/proxy/list', {
      method: 'POST', headers:{ 'Authorization': token, 'Content-Type':'application/json' },
      body: JSON.stringify({}), credentials: 'include'
    });
    if (!resp.ok) throw new Error('API fetch failed: '+resp.status);
    const data = await resp.json();
    await send({type:'setConfig', config:{ mode:'token', token, apiBase, _proxyList:data }});
    renderProxyList(data);
    setStatus('Loaded '+(data.length||0)+' proxies');
  }catch(err){ setError(err.message || String(err)); }
});

document.getElementById('rotateAll').addEventListener('click', async ()=>{
  setError(''); setStatus('Rotating...');
  const token = document.getElementById('token').value.trim();
  const apiBase = document.getElementById('apiBase').value.trim().replace(/\/$/,'');
  if (!token) return setError('Token required');
  try{
    const listResp = await fetch(apiBase + '/server/proxy/list', {
      method:'POST', headers:{ Authorization: token, 'Content-Type':'application/json'}, body: JSON.stringify({})
    });
    const list = await listResp.json();
    for (const p of list){
      try{
        await fetch(apiBase + '/server/proxy/rotate', {
          method:'POST', headers:{ Authorization: token, 'Content-Type':'application/json'},
          body: JSON.stringify({ ip: p.ip || p.host, port: Number(p.port || p.p) })
        });
      }catch(e){}
    }
    setStatus('Rotate requests sent'); document.getElementById('loadList').click();
  }catch(e){ setError(e.message || String(e)); }
});

document.getElementById('enableAll').addEventListener('click', async ()=>{
  const cfg = (await send({type:'getConfig'})).config; cfg.enabled = true;
  await send({type:'setConfig', config: cfg}); await send({type:'applyConfigNow'});
  setStatus('✅ Proxy enabled');
});
document.getElementById('disableAll').addEventListener('click', async ()=>{
  const cfg = (await send({type:'getConfig'})).config; cfg.enabled = false;
  await send({type:'setConfig', config: cfg}); await send({type:'applyConfigNow'});
  setStatus('⛔ Proxy disabled'); updateFooter(null);
});

document.getElementById('openOptions').addEventListener('click', ()=> chrome.runtime.openOptionsPage());

loadConfigToUI();
